<?php
/**
 * Avada Builder Pattern Helper class.
 *
 * @package Avada-Builder
 * @since 2.2
 */

// Do not allow directly accessing this file.
if ( ! defined( 'ABSPATH' ) ) {
	exit( 'Direct script access denied.' );
}

/**
 * Avada Builder Pattern Helper class.
 *
 * @since 2.2
 */
class Fusion_Builder_Pattern_Helper {

	/**
	 * Class constructor.
	 *
	 * @since 2.2
	 * @access public
	 */
	public function __construct() {
	}

	/**
	 * Get pattern params.
	 *
	 * @since 3.8
	 * @access public
	 * @param array $args The placeholder arguments.
	 * @return array
	 */
	public static function get_params( $args ) {
		$selector = isset( $args['selector'] ) ? $args['selector'] : '';

		return [
			[
				'type'        => 'radio_button_set',
				'heading'     => esc_attr__( 'Pattern', 'fusion-builder' ),
				'param_name'  => 'pattern_bg',
				'description' => esc_attr__( 'Select pattern.', 'fusion-builder' ),
				'group'       => esc_attr__( 'Background', 'fusion-builder' ),
				'subgroup'    => [
					'name' => 'background_type',
					'tab'  => 'pattern',
				],
				'default'     => 'none',
				'value'       => [
					'none'         => esc_attr__( 'None', 'fusion-builder' ),
					'abstract'     => esc_attr__( 'Abstract', 'fusion-builder' ),
					'bricks'       => esc_attr__( 'Bricks', 'fusion-builder' ),
					'circles'      => esc_attr__( 'Circles', 'fusion-builder' ),
					'dots'         => esc_attr__( 'Dots', 'fusion-builder' ),
					'grid'         => esc_attr__( 'Grid', 'fusion-builder' ),
					'hexagon'      => esc_attr__( 'Hexagon', 'fusion-builder' ),
					'half-diamond' => esc_attr__( 'Half Diamond', 'fusion-builder' ),
					'half-circle'  => esc_attr__( 'Half Circle', 'fusion-builder' ),
					'square'       => esc_attr__( 'Square', 'fusion-builder' ),
					'square-2'     => esc_attr__( 'Square 2', 'fusion-builder' ),
					'triangle'     => esc_attr__( 'Triangle', 'fusion-builder' ),
					'triangle-2'   => esc_attr__( 'Triangle 2', 'fusion-builder' ),
					'triangle-3'   => esc_attr__( 'Triangle 3', 'fusion-builder' ),
					'pastel'       => esc_attr__( 'Pastel', 'fusion-builder' ),
					'wave'         => esc_attr__( 'Wave', 'fusion-builder' ),
					'x'            => esc_attr__( 'X', 'fusion-builder' ),
					'custom'       => esc_attr__( 'Custom', 'fusion-builder' ),
				],
				'icons'       => [
					'none'         => '<span class="fusiona-minus"></span>',
					'abstract'     => '<span class="fusiona-pattern-abstract"></span>',
					'bricks'       => '<span class="fusiona-pattern-bricks"></span>',
					'circles'      => '<span class="fusiona-pattern-circles"></span>',
					'dots'         => '<span class="fusiona-pattern-dots"></span>',
					'grid'         => '<span class="fusiona-pattern-grid"></span>',
					'hexagon'      => '<span class="fusiona-pattern-hexagon"></span>',
					'half-diamond' => '<span class="fusiona-pattern-half-diamond"></span>',
					'half-circle'  => '<span class="fusiona-pattern-half-circle"></span>',
					'square'       => '<span class="fusiona-pattern-square"></span>',
					'square-2'     => '<span class="fusiona-pattern-square-2"></span>',
					'triangle'     => '<span class="fusiona-pattern-triangle"></span>',
					'triangle-2'   => '<span class="fusiona-pattern-triangle-2"></span>',
					'triangle-3'   => '<span class="fusiona-pattern-triangle-3"></span>',
					'pastel'       => '<span class="fusiona-pattern-pastel"></span>',
					'wave'         => '<span class="fusiona-pattern-wave"></span>',
					'x'            => '<span class="fusiona-pattern-x"></span>',
					'custom'       => '<span class="fusiona-cog"></span>',
				],
				'grid_layout' => true,
				'back_icons'  => true,
				'callback'    => [
					'function' => 'fusion_update_pattern_style',
					'args'     => [
						'selector' => $selector,
					],
				],
			],
			[
				'type'        => 'upload',
				'heading'     => esc_attr__( 'Custom Pattern', 'fusion-builder' ),
				'param_name'  => 'pattern_custom_bg',
				'description' => esc_attr__( 'Upload your custom pattern.', 'fusion-builder' ),
				'group'       => esc_attr__( 'Background', 'fusion-builder' ),
				'subgroup'    => [
					'name' => 'background_type',
					'tab'  => 'pattern',
				],
				'callback'    => [
					'function' => 'fusion_update_pattern_style',
					'args'     => [
						'selector' => $selector,
					],
				],
				'dependency'  => [
					[
						'element'  => 'pattern_bg',
						'value'    => 'custom',
						'operator' => '==',
					],
				],
			],
			[
				'type'        => 'colorpickeralpha',
				'heading'     => esc_attr__( 'Pattern Color', 'fusion-builder' ),
				'param_name'  => 'pattern_bg_color',
				'description' => esc_attr__( 'Set the pattern color.', 'fusion-builder' ),
				'group'       => esc_attr__( 'Background', 'fusion-builder' ),
				'default'     => 'rgba(0,0,0,0.3)',
				'subgroup'    => [
					'name' => 'background_type',
					'tab'  => 'pattern',
				],
				'callback'    => [
					'function' => 'fusion_update_pattern_style',
					'args'     => [
						'selector' => $selector,
					],
				],
				'dependency'  => [
					[
						'element'  => 'pattern_bg',
						'value'    => 'none',
						'operator' => '!=',
					],
					[
						'element'  => 'pattern_bg',
						'value'    => 'custom',
						'operator' => '!=',
					],
				],
			],
			[
				'type'        => 'radio_button_set',
				'heading'     => esc_attr__( 'Pattern Style', 'fusion-builder' ),
				'param_name'  => 'pattern_bg_style',
				'description' => esc_attr__( 'Select the pattern style.', 'fusion-builder' ),
				'group'       => esc_attr__( 'Background', 'fusion-builder' ),
				'default'     => 'default',
				'value'       => [
					'default'  => esc_attr__( 'Default', 'fusion-builder' ),
					'inverted' => esc_attr__( 'Inverted', 'fusion-builder' ),
				],
				'subgroup'    => [
					'name' => 'background_type',
					'tab'  => 'pattern',
				],
				'callback'    => [
					'function' => 'fusion_update_pattern_style',
					'args'     => [
						'selector' => $selector,
					],
				],
				'dependency'  => [
					[
						'element'  => 'pattern_bg',
						'value'    => 'none',
						'operator' => '!=',
					],
					[
						'element'  => 'pattern_bg',
						'value'    => 'custom',
						'operator' => '!=',
					],
				],
			],
			[
				'type'        => 'range',
				'heading'     => esc_attr__( 'Pattern Opacity', 'fusion-builder' ),
				'param_name'  => 'pattern_bg_opacity',
				'description' => esc_attr__( 'Set the pattern opacity.', 'fusion-builder' ),
				'min'         => 0,
				'max'         => 100,
				'value'       => 100,
				'step'        => 1,
				'group'       => esc_attr__( 'Background', 'fusion-builder' ),
				'subgroup'    => [
					'name' => 'background_type',
					'tab'  => 'pattern',
				],
				'callback'    => [
					'function' => 'fusion_update_pattern_style',
					'args'     => [
						'selector' => $selector,
					],
				],
				'dependency'  => [
					[
						'element'  => 'pattern_bg',
						'value'    => 'none',
						'operator' => '!=',
					],
				],
			],
			[
				'type'        => 'textfield',
				'heading'     => esc_attr__( 'Pattern Size', 'fusion-builder' ),
				'param_name'  => 'pattern_bg_size',
				'description' => esc_attr__( 'Set the pattern size. Enter values including any valid CSS unit, ex: 20%.', 'fusion-builder' ),
				'group'       => esc_attr__( 'Background', 'fusion-builder' ),
				'subgroup'    => [
					'name' => 'background_type',
					'tab'  => 'pattern',
				],
				'callback'    => [
					'function' => 'fusion_update_pattern_style',
					'args'     => [
						'selector' => $selector,
					],
				],
				'dependency'  => [
					[
						'element'  => 'pattern_bg',
						'value'    => 'none',
						'operator' => '!=',
					],
				],
			],
			[
				'type'        => 'select',
				'heading'     => esc_attr__( 'Pattern Blend Mode', 'fusion-builder' ),
				'param_name'  => 'pattern_bg_blend_mode',
				'description' => esc_attr__( 'Set how pattern will blend with element background.', 'fusion-builder' ),
				'group'       => esc_attr__( 'Background', 'fusion-builder' ),
				'value'       => [
					'normal'       => esc_attr__( 'Normal', 'fusion-builder' ),
					'multiply'     => esc_attr__( 'Multiply', 'fusion-builder' ),
					'screen'       => esc_attr__( 'Screen', 'fusion-builder' ),
					'overlay'      => esc_attr__( 'Overlay', 'fusion-builder' ),
					'darken'       => esc_attr__( 'Darken', 'fusion-builder' ),
					'lighten'      => esc_attr__( 'Lighten', 'fusion-builder' ),
					'color-dodge'  => esc_attr__( 'Color Dodge', 'fusion-builder' ),
					'color-burn'   => esc_attr__( 'Color Burn', 'fusion-builder' ),
					'hard-light'   => esc_attr__( 'Hard Light', 'fusion-builder' ),
					'soft-light'   => esc_attr__( 'Soft Light', 'fusion-builder' ),
					'difference'   => esc_attr__( 'Difference', 'fusion-builder' ),
					'exclusion'    => esc_attr__( 'Exclusion', 'fusion-builder' ),
					'hue'          => esc_attr__( 'Hue', 'fusion-builder' ),
					'saturation'   => esc_attr__( 'Saturation', 'fusion-builder' ),
					'color'        => esc_attr__( 'Color', 'fusion-builder' ),
					'luminosity'   => esc_attr__( 'Luminosity', 'fusion-builder' ),
					'plus-lighter' => esc_attr__( 'Plus Lighter', 'fusion-builder' ),
				],
				'default'     => 'normal',
				'subgroup'    => [
					'name' => 'background_type',
					'tab'  => 'pattern',
				],
				'callback'    => [
					'function' => 'fusion_update_pattern_style',
					'args'     => [
						'selector' => $selector,
					],
				],
				'dependency'  => [
					[
						'element'  => 'pattern_bg',
						'value'    => 'none',
						'operator' => '!=',
					],
				],
			],
		];
	}

	/**
	 * Get pattern element.
	 *
	 * @since 3.8
	 * @access public
	 * @param array $atts The element attributes.
	 * @return array
	 */
	public static function get_element( $atts ) {

		// Early exit if no pattern selected.
		if ( '' === $atts['pattern_bg'] || 'none' === $atts['pattern_bg'] ) {
			return;
		}

		$style = '';
		if ( 'custom' === $atts['pattern_bg'] ) {
			$style .= 'background-image:  url(' . $atts['pattern_custom_bg'] . ');';
		} else {
			$style .= 'background-image:  url(data:image/svg+xml;utf8,' . rawurlencode( self::get_pattern( $atts['pattern_bg'], $atts['pattern_bg_color'], $atts['pattern_bg_style'] ) ) . ');';
		}

		if ( '' !== $atts['pattern_bg_opacity'] ) {
			$style .= 'opacity: ' . ( intval( $atts['pattern_bg_opacity'] ) / 100 ) . ' ;';
		}
		if ( '' !== $atts['pattern_bg_size'] ) {
			$style .= 'background-size:' . $atts['pattern_bg_size'] . ';';
		}
		if ( '' !== $atts['pattern_bg_blend_mode'] ) {
			$style .= 'mix-blend-mode:' . $atts['pattern_bg_blend_mode'] . ';';
		}
		$element = '<div class="awb-background-pattern" style="' . esc_attr( $style ) . '"></div>';

		return $element;
	}

	/**
	 * Get pattern.
	 *
	 * @since 3.8
	 * @access public
	 * @param string $name  The mask name.
	 * @param string $color The mask color.
	 * @param string $style The mask style.
	 * @return array
	 */
	public static function get_pattern( $name, $color = '', $style = '' ) {
		$style = '' === $style ? 'default' : $style;
		$color = $color ? Fusion_Color::new_color( $color )->toCss( 'rgba' ) : 'rgba(0,0,0,0.3)';

		$patterns = [
			'abstract'     => [
				'default'  => '<svg width="120" height="120" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#prefix__clip0_44_400)" fill="' . $color . '"><path d="M4.5 61.92A1.08 1.08 0 003.42 63H4.5v-1.08zM.9 60.48c.375 0 .704-.19.898-.48H.002c.194.29.523.48.898.48zM18.9 61.92A1.08 1.08 0 0017.82 63h1.08v-1.08zM11.7 61.92A1.08 1.08 0 0010.62 63h2.16a1.08 1.08 0 00-1.08-1.08zM5.58 63a1.08 1.08 0 00-1.08-1.08V63h1.08zM15.3 60.48c.375 0 .703-.19.898-.48h-1.796c.194.29.523.48.898.48zM8.1 60.48c.375 0 .704-.19.898-.48H7.202c.194.29.523.48.898.48zM33.3 63v-1.08A1.08 1.08 0 0032.22 63h1.08zM26.1 61.92A1.08 1.08 0 0025.02 63h2.16a1.08 1.08 0 00-1.08-1.08zM19.98 63a1.08 1.08 0 00-1.08-1.08V63h1.08zM29.7 60.48c.375 0 .704-.19.898-.48h-1.796c.194.29.523.48.898.48zM22.5 60.48c.375 0 .703-.19.898-.48h-1.796c.194.29.523.48.898.48zM46.62 63h1.08v-1.08A1.08 1.08 0 0046.62 63zM40.5 61.92A1.08 1.08 0 0039.42 63h2.16a1.08 1.08 0 00-1.08-1.08zM34.38 63a1.08 1.08 0 00-1.08-1.08V63h1.08zM44.1 60.48c.375 0 .705-.19.898-.48h-1.796c.195.29.523.48.898.48zM36.9 60.48c.375 0 .705-.19.898-.48h-1.796c.193.29.523.48.898.48zM55.98 63a1.08 1.08 0 00-2.16 0h2.16zM48.78 63a1.08 1.08 0 00-1.08-1.08V63h1.08zM58.5 60.48c.375 0 .704-.19.898-.48h-1.796c.195.29.523.48.898.48zM51.3 60.48c.375 0 .705-.19.898-.48h-1.796c.193.29.523.48.898.48zM4.5 76.32a1.08 1.08 0 00-1.08 1.08H4.5v-1.08zM3.42 70.2c0 .596.484 1.08 1.08 1.08v-2.16a1.08 1.08 0 00-1.08 1.08zM3.42 63c0 .596.484 1.08 1.08 1.08V63H3.42zM1.98 66.6a1.08 1.08 0 00-1.707-.879c.06.627.125 1.252.204 1.872a1.08 1.08 0 001.504-.995v.002zM1.38 72.834c.118.543.245 1.084.377 1.621a1.075 1.075 0 00-.377-1.621zM18.9 76.32a1.08 1.08 0 00-1.08 1.08h1.08v-1.08zM17.82 70.2c0 .596.483 1.08 1.08 1.08v-2.16a1.08 1.08 0 00-1.08 1.08zM17.82 63c0 .596.483 1.08 1.08 1.08V63h-1.08zM11.7 76.32a1.08 1.08 0 00-1.08 1.08h2.16a1.08 1.08 0 00-1.08-1.08zM12.78 70.2a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zM11.7 64.08A1.08 1.08 0 0012.78 63h-2.16c0 .596.484 1.08 1.08 1.08zM4.5 76.32v1.08h1.08a1.08 1.08 0 00-1.08-1.08zM5.58 70.2a1.08 1.08 0 00-1.08-1.08v2.16a1.08 1.08 0 001.08-1.08zM5.58 63H4.5v1.08A1.08 1.08 0 005.58 63zM16.38 66.6a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zM16.38 73.8a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zM9.18 66.6a1.08 1.08 0 10-2.161 0 1.08 1.08 0 002.161 0zM9.18 73.8a1.08 1.08 0 10-2.161 0 1.08 1.08 0 002.161 0zM33.3 77.4v-1.08a1.08 1.08 0 00-1.08 1.08h1.08zM32.22 70.2c0 .596.483 1.08 1.08 1.08v-2.16a1.08 1.08 0 00-1.08 1.08zM33.3 64.08V63h-1.08c0 .596.483 1.08 1.08 1.08zM26.1 76.32a1.08 1.08 0 00-1.08 1.08h2.16a1.08 1.08 0 00-1.08-1.08zM27.18 70.2a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zM26.1 64.08A1.08 1.08 0 0027.18 63h-2.16c0 .596.483 1.08 1.08 1.08zM18.9 76.32v1.08h1.08a1.08 1.08 0 00-1.08-1.08zM19.98 70.2a1.08 1.08 0 00-1.08-1.08v2.16a1.08 1.08 0 001.08-1.08zM19.98 63H18.9v1.08A1.08 1.08 0 0019.98 63zM30.78 66.6a1.08 1.08 0 10-2.161 0 1.08 1.08 0 002.161 0zM30.78 73.8a1.08 1.08 0 10-2.161 0 1.08 1.08 0 002.161 0zM23.58 66.6a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zM23.58 73.8a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zM46.62 77.4h1.08v-1.08a1.081 1.081 0 00-1.08 1.08zM46.62 70.2a1.08 1.08 0 001.08 1.08v-2.16a1.081 1.081 0 00-1.08 1.08zM46.62 63a1.081 1.081 0 001.08 1.08V63h-1.08zM40.5 76.32a1.08 1.08 0 00-1.08 1.08h2.16a1.08 1.08 0 00-1.08-1.08zM41.58 70.2a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zM40.5 64.08A1.08 1.08 0 0041.58 63h-2.16c0 .596.483 1.08 1.08 1.08zM34.38 77.4a1.08 1.08 0 00-1.08-1.08v1.08h1.08zM34.38 70.2a1.08 1.08 0 00-1.08-1.08v2.16a1.08 1.08 0 001.08-1.08zM33.3 64.08A1.08 1.08 0 0034.38 63H33.3v1.08zM43.02 66.6a1.08 1.08 0 102.161 0 1.08 1.08 0 00-2.161 0zM43.02 73.8a1.08 1.08 0 102.161 0 1.08 1.08 0 00-2.161 0zM35.82 66.6a1.08 1.08 0 102.16 0 1.08 1.08 0 00-2.16 0zM35.82 73.8a1.08 1.08 0 102.161 0 1.08 1.08 0 00-2.161 0zM54.9 76.32a1.08 1.08 0 00-1.08 1.08h2.16a1.08 1.08 0 00-1.08-1.08zM54.9 69.12a1.08 1.08 0 100 2.161 1.08 1.08 0 000-2.161zM54.9 64.08A1.08 1.08 0 0055.98 63h-2.16c0 .596.484 1.08 1.08 1.08zM48.78 77.4a1.08 1.08 0 00-1.08-1.08v1.08h1.08zM48.78 70.2a1.08 1.08 0 00-1.08-1.08v2.16a1.08 1.08 0 001.08-1.08zM47.7 64.08A1.08 1.08 0 0048.78 63H47.7v1.08zM57.42 66.6a1.08 1.08 0 102.161 0 1.08 1.08 0 00-2.161 0zM57.42 73.8a1.08 1.08 0 102.161 0 1.08 1.08 0 00-2.161 0zM52.38 66.6a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zM50.22 73.8a1.08 1.08 0 102.161 0 1.08 1.08 0 00-2.161 0zM3.42 77.4c0 .596.484 1.08 1.08 1.08V77.4H3.42zM18.9 90.72a1.08 1.08 0 00-1.08 1.08h1.08v-1.08zM17.82 84.6c0 .597.483 1.08 1.08 1.08v-2.16a1.08 1.08 0 00-1.08 1.08zM17.82 77.4c0 .596.483 1.08 1.08 1.08V77.4h-1.08zM11.7 90.72a1.08 1.08 0 00-1.08 1.08h2.16a1.08 1.08 0 00-1.08-1.08zM12.78 84.6a1.08 1.08 0 10-2.161 0 1.08 1.08 0 002.161 0zM11.7 78.48a1.08 1.08 0 001.08-1.08h-2.16c0 .596.484 1.08 1.08 1.08zM4.807 83.564c.216.504.44 1.004.668 1.5a1.082 1.082 0 00-.668-1.5zM4.5 78.48a1.08 1.08 0 001.08-1.08H4.5v1.08zM16.38 81a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zM16.38 88.2a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zM9.18 81a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zM9.18 88.2a1.08 1.08 0 00-2.159-.02c.168.315.338.625.51.936.165.104.36.165.569.165a1.08 1.08 0 001.08-1.08zM33.3 91.8v-1.08a1.08 1.08 0 00-1.08 1.08h1.08zM32.22 84.6c0 .597.483 1.08 1.08 1.08v-2.16a1.08 1.08 0 00-1.08 1.08zM32.22 77.4c0 .596.483 1.08 1.08 1.08V77.4h-1.08zM26.1 90.72a1.08 1.08 0 00-1.08 1.08h2.16a1.08 1.08 0 00-1.08-1.08zM27.18 84.6a1.08 1.08 0 10-2.161 0 1.08 1.08 0 002.161 0zM26.1 78.48a1.08 1.08 0 001.08-1.08h-2.16c0 .596.483 1.08 1.08 1.08zM18.9 90.72v1.08h1.08a1.08 1.08 0 00-1.08-1.08zM19.98 84.6a1.08 1.08 0 00-1.08-1.08v2.16a1.08 1.08 0 001.08-1.08zM19.98 77.4H18.9v1.08a1.08 1.08 0 001.08-1.08zM30.78 81a1.08 1.08 0 10-2.161 0 1.08 1.08 0 002.161 0zM30.78 88.2a1.08 1.08 0 10-2.161 0 1.08 1.08 0 002.161 0zM23.58 81a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zM23.58 88.2a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zM46.62 91.8h1.08v-1.08a1.081 1.081 0 00-1.08 1.08zM46.62 84.6a1.081 1.081 0 001.08 1.08v-2.16a1.08 1.08 0 00-1.08 1.08zM46.62 77.4a1.08 1.08 0 001.08 1.08V77.4h-1.08zM40.5 90.72a1.08 1.08 0 00-1.08 1.08h2.16a1.08 1.08 0 00-1.08-1.08zM39.42 84.6a1.08 1.08 0 102.16 0 1.08 1.08 0 00-2.16 0zM40.5 78.48a1.08 1.08 0 001.08-1.08h-2.16c0 .596.483 1.08 1.08 1.08zM33.3 90.72v1.08h1.08a1.08 1.08 0 00-1.08-1.08zM34.38 84.6a1.08 1.08 0 00-1.08-1.08v2.16a1.08 1.08 0 001.08-1.08zM34.38 77.4H33.3v1.08a1.08 1.08 0 001.08-1.08zM45.18 81a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zM43.02 88.2a1.08 1.08 0 102.161 0 1.08 1.08 0 00-2.161 0zM35.82 81a1.08 1.08 0 102.16 0 1.08 1.08 0 00-2.16 0zM35.82 88.2a1.08 1.08 0 102.16 0 1.08 1.08 0 00-2.16 0zM54.9 90.72a1.08 1.08 0 00-1.08 1.08h2.16a1.08 1.08 0 00-1.08-1.08zM53.82 84.6a1.08 1.08 0 102.161 0 1.08 1.08 0 00-2.161 0zM54.9 78.48a1.08 1.08 0 001.08-1.08h-2.16c0 .596.484 1.08 1.08 1.08zM48.78 91.8a1.08 1.08 0 00-1.08-1.08v1.08h1.08zM48.78 84.6a1.08 1.08 0 00-1.08-1.08v2.16a1.08 1.08 0 001.08-1.08zM47.7 78.48a1.08 1.08 0 001.08-1.08H47.7v1.08zM59.58 81a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zM57.42 88.2a1.08 1.08 0 102.161 0 1.08 1.08 0 00-2.161 0zM50.22 81a1.08 1.08 0 102.16 0 1.08 1.08 0 00-2.16 0zM50.22 88.2a1.08 1.08 0 102.16 0 1.08 1.08 0 00-2.16 0zM17.82 99c0 .596.483 1.08 1.08 1.08v-2.16A1.08 1.08 0 0017.82 99zM17.82 91.8c0 .596.483 1.08 1.08 1.08V91.8h-1.08zM11.7 92.88a1.08 1.08 0 001.08-1.08h-2.16c0 .596.484 1.08 1.08 1.08zM16.38 95.4a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zM33.3 106.2v-1.08a1.08 1.08 0 00-1.08 1.08h1.08zM32.22 99c0 .596.483 1.08 1.08 1.08v-2.16A1.08 1.08 0 0032.22 99zM32.22 91.8c0 .596.483 1.08 1.08 1.08V91.8h-1.08zM26.1 105.12a1.08 1.08 0 00-1.08 1.08h2.16a1.08 1.08 0 00-1.08-1.08zM27.18 99a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zM26.1 92.88a1.08 1.08 0 001.08-1.08h-2.16c0 .596.483 1.08 1.08 1.08zM19.98 99a1.08 1.08 0 00-1.08-1.08v2.16A1.08 1.08 0 0019.98 99zM18.9 92.88a1.08 1.08 0 001.08-1.08H18.9v1.08zM30.78 95.4a1.08 1.08 0 10-2.161 0 1.08 1.08 0 002.161 0zM30.78 102.6a1.08 1.08 0 10-2.161.001 1.08 1.08 0 002.161-.001zM23.58 95.4a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zM23.58 102.6a1.08 1.08 0 10-2.16.001 1.08 1.08 0 002.16-.001zM47.7 106.2v-1.08a1.08 1.08 0 00-1.08 1.08h1.08zM46.62 99a1.08 1.08 0 001.08 1.08v-2.16A1.08 1.08 0 0046.62 99zM46.62 91.8a1.08 1.08 0 001.08 1.08V91.8h-1.08zM40.5 105.12a1.08 1.08 0 00-1.08 1.08h2.16a1.08 1.08 0 00-1.08-1.08zM41.58 99a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zM40.5 92.88a1.08 1.08 0 001.08-1.08h-2.16c0 .596.483 1.08 1.08 1.08zM33.3 105.12v1.08h1.08a1.08 1.08 0 00-1.08-1.08zM34.38 99a1.08 1.08 0 00-1.08-1.08v2.16A1.08 1.08 0 0034.38 99zM34.38 91.8H33.3v1.08a1.08 1.08 0 001.08-1.08zM45.18 95.4a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zM45.18 102.6a1.08 1.08 0 10-2.16.001 1.08 1.08 0 002.16-.001zM37.98 95.4a1.08 1.08 0 10-2.161 0 1.08 1.08 0 002.161 0zM37.98 102.6a1.08 1.08 0 10-2.16.001 1.08 1.08 0 002.16-.001zM54.9 105.12a1.08 1.08 0 00-1.08 1.08h2.16a1.08 1.08 0 00-1.08-1.08zM53.82 99a1.08 1.08 0 102.16 0 1.08 1.08 0 00-2.16 0zM54.9 92.88a1.08 1.08 0 001.08-1.08h-2.16c0 .596.484 1.08 1.08 1.08zM47.7 105.12v1.08h1.08a1.08 1.08 0 00-1.08-1.08zM48.78 99a1.08 1.08 0 00-1.08-1.08v2.16A1.08 1.08 0 0048.78 99zM47.7 92.88a1.08 1.08 0 001.08-1.08H47.7v1.08zM57.42 95.4a1.08 1.08 0 102.161 0 1.08 1.08 0 00-2.161 0zM57.42 102.6a1.08 1.08 0 102.161-.001 1.08 1.08 0 00-2.161.001zM50.22 95.4a1.08 1.08 0 102.16 0 1.08 1.08 0 00-2.16 0zM50.22 102.6a1.08 1.08 0 102.16-.001 1.08 1.08 0 00-2.16.001zM32.24 113.2c.351.184.705.364 1.06.541v-1.421c-.529 0-.968.38-1.06.88zM32.22 106.2c0 .596.483 1.08 1.08 1.08v-1.08h-1.08zM26.1 107.28a1.08 1.08 0 001.08-1.08h-2.16c0 .596.483 1.08 1.08 1.08zM30.78 109.8a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zM46.62 113.4a1.08 1.08 0 001.08 1.08v-2.16a1.08 1.08 0 00-1.08 1.08zM46.62 106.2a1.08 1.08 0 001.08 1.08v-1.08h-1.08zM41.58 113.4a1.08 1.08 0 10-2.161.001 1.08 1.08 0 002.161-.001zM40.5 107.28a1.08 1.08 0 001.08-1.08h-2.16c0 .596.483 1.08 1.08 1.08z"/><path d="M33.3 112.32v1.421c.264.13.527.263.793.391a1.08 1.08 0 00-.793-1.812zM33.3 107.28a1.08 1.08 0 001.08-1.08H33.3v1.08zM45.18 109.8a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zM45.18 117a1.08 1.08 0 10-1.966.616c.441.129.882.25 1.327.37a1.08 1.08 0 00.64-.986zM37.98 109.8a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zM54.9 119.52c-.236 0-.455.076-.632.205.466.045.934.084 1.403.118a1.073 1.073 0 00-.77-.323H54.9zM53.82 113.4a1.08 1.08 0 102.16-.001 1.08 1.08 0 00-2.16.001zM54.9 107.28a1.08 1.08 0 001.08-1.08h-2.16c0 .596.483 1.08 1.08 1.08zM48.78 113.4a1.08 1.08 0 00-1.08-1.08v2.16a1.08 1.08 0 001.08-1.08zM47.7 107.28a1.08 1.08 0 001.08-1.08H47.7v1.08zM59.58 109.8a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zM59.58 117a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zM50.22 109.8a1.08 1.08 0 102.16-.001 1.08 1.08 0 00-2.16.001zM50.22 117a1.08 1.08 0 102.16 0 1.08 1.08 0 00-2.16 0zM74.341 19.8h.72V1.907c-.24-.062-.48-.118-.72-.177v18.072-.002zM67.141 19.8h.72V.512c-.24-.032-.48-.059-.72-.087V19.8zM60 19.8h.66V.009c-.219-.002-.439-.01-.66-.01v19.8zM70.741 19.8h.72V1.098c-.24-.046-.479-.095-.72-.137V19.8zM63.541 19.8h.72V.152l-.72-.047V19.8zM95.941 19.8h.72v-7.296c-.24-.186-.479-.372-.72-.552V19.8zM88.741 19.8h.72V7.723c-.24-.135-.479-.271-.72-.401v12.48-.002zM81.541 19.8h.72V4.268c-.24-.097-.48-.19-.72-.28V19.8zM92.341 19.8h.72V9.927c-.24-.16-.479-.32-.72-.473V19.8zM85.141 19.8h.72V5.847c-.24-.115-.48-.225-.72-.336V19.8zM77.941 19.8h.72V2.962c-.24-.078-.479-.159-.72-.234V19.8zM103.141 19.8h.72v-.736a62.632 62.632 0 00-.72-.759V19.8zM99.541 19.8h.72v-4.286c-.238-.216-.479-.427-.72-.64V19.8z"/><path d="M74.341 41.325v.075h.72V19.8h-.72v21.525zM67.141 41.325v.075h.72V19.8h-.72v21.525zM60.66 41.325V19.8H60v21.6h.66v-.075zM70.741 41.325v.075h.72V19.8h-.72v21.525zM63.541 41.325v.075h.72V19.8h-.72v21.525zM95.941 41.325v.075h.72V19.8h-.72v21.525zM88.741 41.325v.075h.72V19.8h-.72v21.525zM81.541 22.05V41.4h.72V19.8h-.72v2.25zM92.341 41.325v.075h.72V19.8h-.72v21.525zM85.141 41.325v.075h.72V19.8h-.72v21.525zM77.941 22.05V41.4h.72V19.8h-.72v2.25zM110.341 41.4h.72V28.482c-.236-.382-.475-.76-.72-1.136V41.4zM103.141 22.05V41.4h.72V19.8h-.72v2.25zM113.941 41.4h.72v-6.177a57.484 57.484 0 00-.72-1.523v7.7zM106.741 41.4h.72V23.293a73.002 73.002 0 00-.72-.911V41.4zM99.541 22.05V41.4h.72V19.8h-.72v2.25z"/><path d="M75.06 41.4h-.719V60h.72V41.4zM67.86 41.4h-.719V60h.72V41.4zM60.66 43.65V41.4H60V60h.66V43.65zM71.46 41.4h-.719V60h.72V41.4zM64.26 41.4h-.719V60h.72V41.4zM96.66 41.4h-.719V60h.72V41.4zM89.46 41.4h-.719V60h.72V41.4zM82.26 43.65V41.4h-.719V60h.72V43.65zM93.06 41.4h-.719V60h.72V41.4zM85.86 41.4h-.719V60h.72V41.4zM78.66 43.65V41.4h-.719V60h.72V43.65zM118.261 60V45.616a59.776 59.776 0 00-.72-2.659V60h.72zM111.061 41.4h-.72V60h.72V41.4zM103.861 43.65V41.4h-.72V60h.72V43.65zM114.661 41.4h-.72V60h.72V41.4zM107.461 41.4h-.72V60h.72V41.4zM100.261 43.65V41.4h-.72V60h.72V43.65zM120 120c-33.138 0-60-26.862-60-60h60v60zM60 60H0V0c33.138 0 60 26.864 60 60z"/></g><defs><clipPath id="prefix__clip0_44_400"><path fill="#fff" d="M0 0h120v120H0z"/></clipPath></defs></svg>',
				'inverted' => '<svg width="120" height="120" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#prefix__clip0_44_635)"><path fill-rule="evenodd" clip-rule="evenodd" d="M0 0h60V60h.66V.009c-.093 0-.187-.003-.281-.004h-.006A20.234 20.234 0 0060.009 0H120v120H0V0zm67.141 19.8V.425l.19.022c.177.021.354.042.53.065V60h-.72V19.8zm7.2 0V1.73l.25.061h.002c.156.038.312.076.468.116V60h-.72V19.8zm-2.88-18.702V60h-.72V.96c.241.044.48.092.72.138zM64.26 19.8V.152l-.72-.047V60h.72V19.8zm32.4-7.296V60h-.72V11.952c.241.18.48.366.72.552zM82.26 19.8V4.268c-.24-.097-.48-.19-.72-.28V60h.72V19.8zm10.8-9.873V60h-.72V9.454c.241.153.48.314.72.473zm-7.2-4.08V60h-.72V5.51l.115.054c.202.093.404.187.605.283zm-7.2-2.885V60h-.72V2.729c.185.057.369.118.552.178l.168.055zm24.48 15.343V60h.72V19.064a61.91 61.91 0 00-.72-.759zm-3.6 1.495v-4.925l.124.11c.2.175.399.35.596.53V60h-.72V19.8zm11.52 21.6V28.482c-.236-.382-.475-.76-.72-1.136V60h.72V41.4zm3.6 0v-6.177a57.484 57.484 0 00-.72-1.523V60h.72V41.4zm-7.2-18.107V60h-.72V22.382c.243.302.482.605.72.91zM89.461 60V7.723a41.03 41.03 0 00-.72-.401V60h.72zm28.8-14.384V60h-.72V42.957c.259.877.5 1.764.72 2.66zM.9 60.48c.375 0 .704-.19.898-.48H.002c.194.29.523.48.898.48zm3.6 1.44a1.08 1.08 0 100 2.16 1.08 1.08 0 000-2.16zM17.82 63a1.08 1.08 0 112.16 0 1.08 1.08 0 01-2.16 0zm-6.12-1.08a1.08 1.08 0 100 2.16 1.08 1.08 0 000-2.16zm3.6-1.44c.375 0 .703-.19.898-.48h-1.796c.194.29.523.48.898.48zM8.998 60a1.08 1.08 0 01-1.796 0h1.796zM33.3 61.92a1.08 1.08 0 100 2.16 1.08 1.08 0 000-2.16zM25.02 63a1.08 1.08 0 112.16 0 1.08 1.08 0 01-2.16 0zm5.578-3a1.08 1.08 0 01-1.796 0h1.796zm-8.098.48c.375 0 .703-.19.898-.48h-1.796c.194.29.523.48.898.48zm25.2 1.44a1.08 1.08 0 110 2.16 1.08 1.08 0 010-2.16zm-7.2 0a1.08 1.08 0 100 2.16 1.08 1.08 0 000-2.16zm3.6-1.44c.375 0 .705-.19.898-.48h-1.796c.194.29.523.48.898.48zM37.798 60a1.078 1.078 0 01-1.796 0h1.796zm18.182 3a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zm2.52-2.52c.375 0 .703-.19.898-.48h-1.796c.194.29.523.48.898.48zM52.198 60a1.078 1.078 0 01-1.796 0h1.796zM4.5 76.32a1.08 1.08 0 100 2.16 1.08 1.08 0 000-2.16zm0-5.04a1.08 1.08 0 110-2.161 1.08 1.08 0 010 2.161zM.9 65.52c.596 0 1.08.483 1.08 1.08v-.002a1.08 1.08 0 01-1.503.995 58.2 58.2 0 01-.204-1.872c.177-.127.393-.202.627-.202zm.48 7.314c.118.543.245 1.084.377 1.621a1.075 1.075 0 00-.377-1.621zM17.82 77.4a1.08 1.08 0 112.16 0 1.08 1.08 0 01-2.16 0zm0-7.2a1.08 1.08 0 102.16 0 1.08 1.08 0 00-2.16 0zm-6.12 6.12a1.08 1.08 0 100 2.16 1.08 1.08 0 000-2.16zm0-7.2a1.08 1.08 0 110 2.16 1.08 1.08 0 010-2.16zm4.68-2.52a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zm-1.08 6.12a1.08 1.08 0 110 2.16 1.08 1.08 0 010-2.16zM9.18 66.6a1.08 1.08 0 10-2.161 0 1.08 1.08 0 002.161 0zM8.1 72.72a1.08 1.08 0 110 2.16 1.08 1.08 0 010-2.16zm25.2 3.6a1.08 1.08 0 100 2.16 1.08 1.08 0 000-2.16zm0-5.04a1.08 1.08 0 110-2.161 1.08 1.08 0 010 2.161zm-8.28 6.12a1.08 1.08 0 112.16 0 1.08 1.08 0 01-2.16 0zm2.16-7.2a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zm2.52-4.68a1.08 1.08 0 110 2.16 1.08 1.08 0 010-2.16zm1.08 8.28a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zm-8.28-8.28a1.08 1.08 0 110 2.16 1.08 1.08 0 010-2.16zm1.08 8.28a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zm23.04 3.6a1.08 1.08 0 112.161 0 1.08 1.08 0 01-2.161 0zm0-7.2a1.081 1.081 0 102.162 0 1.081 1.081 0 00-2.162 0zm-6.12 6.12a1.08 1.08 0 100 2.16 1.08 1.08 0 000-2.16zm0-7.2a1.08 1.08 0 110 2.16 1.08 1.08 0 010-2.16zm2.52-2.52a1.08 1.08 0 102.161 0 1.08 1.08 0 00-2.161 0zm1.08 8.28a1.08 1.08 0 110-2.161 1.08 1.08 0 010 2.161zm-8.28-8.28a1.08 1.08 0 102.161 0 1.08 1.08 0 00-2.161 0zm1.08 8.28a1.08 1.08 0 110-2.161 1.08 1.08 0 010 2.161zm18 1.44a1.08 1.08 0 100 2.16 1.08 1.08 0 000-2.16zm-1.08-6.12a1.08 1.08 0 112.16 0 1.08 1.08 0 01-2.16 0zm3.6-3.6a1.08 1.08 0 102.16 0 1.08 1.08 0 00-2.16 0zm1.08 8.28a1.08 1.08 0 110-2.161 1.08 1.08 0 010 2.161zm-6.12-8.28a1.08 1.08 0 10-2.16 0 1.08 1.08 0 002.16 0zm-1.08 8.28a1.08 1.08 0 110-2.161 1.08 1.08 0 010 2.161zM17.82 91.8a1.08 1.08 0 112.161 0 1.08 1.08 0 01-2.161 0zm0-7.2a1.08 1.08 0 102.16 0 1.08 1.08 0 00-2.16 0zm-6.12 6.12a1.08 1.08 0 100 2.161 1.08 1.08 0 000-2.161zm0-7.2a1.08 1.08 0 110 2.161 1.08 1.08 0 010-2.161zm-6.225 1.544a62.995 62.995 0 01-.668-1.5 1.082 1.082 0 01.668 1.5zM15.3 79.92a1.08 1.08 0 110 2.161 1.08 1.08 0 010-2.161zm1.08 8.28a1.08 1.08 0 10-2.161 0 1.08 1.08 0 002.161 0zM8.1 79.92a1.08 1.08 0 110 2.161 1.08 1.08 0 010-2.161zm1.08 8.28a1.08 1.08 0 00-2.159-.02c.168.315.338.625.51.936a1.08 1.08 0 001.65-.916zm23.04 3.6a1.08 1.08 0 112.161 0 1.08 1.08 0 01-2.161 0zm0-7.2a1.08 1.08 0 102.161 0 1.08 1.08 0 00-2.161 0zm-6.12 6.12a1.08 1.08 0 100 2.161 1.08 1.08 0 000-2.161zm0-7.2a1.08 1.08 0 110 2.161 1.08 1.08 0 010-2.161zM30.78 81a1.08 1.08 0 10-2.161 0 1.08 1.08 0 002.161 0zm-1.08 6.12a1.08 1.08 0 110 2.161 1.08 1.08 0 010-2.161zM23.58 81a1.08 1.08 0 10-2.161 0 1.08 1.08 0 002.161 0zm-1.08 6.12a1.08 1.08 0 110 2.161 1.08 1.08 0 010-2.161zm25.09 3.605a1.08 1.08 0 10.222 2.15 1.08 1.08 0 00-.223-2.15zm0-5.05a1.081 1.081 0 11.222-2.15 1.081 1.081 0 01-.223 2.15zM39.42 91.8a1.08 1.08 0 112.161 0 1.08 1.08 0 01-2.161 0zm0-7.2a1.08 1.08 0 102.161 0 1.08 1.08 0 00-2.161 0zm4.68-4.68a1.08 1.08 0 110 2.161 1.08 1.08 0 010-2.161zm-1.08 8.28a1.08 1.08 0 102.161 0 1.08 1.08 0 00-2.161 0zm-6.12-6.12a1.08 1.08 0 110-2.161 1.08 1.08 0 010 2.161zm-1.08 6.12a1.08 1.08 0 102.161 0 1.08 1.08 0 00-2.161 0zm18 3.6a1.08 1.08 0 112.161 0 1.08 1.08 0 01-2.161 0zm0-7.2a1.08 1.08 0 102.16 0 1.08 1.08 0 00-2.16 0zm4.68-4.68a1.08 1.08 0 110 2.161 1.08 1.08 0 010-2.161zm-1.08 8.28a1.08 1.08 0 102.16 0 1.08 1.08 0 00-2.16 0zm-6.12-6.12a1.08 1.08 0 110-2.16 1.08 1.08 0 010 2.16zm-1.08 6.12a1.08 1.08 0 102.16 0 1.08 1.08 0 00-2.16 0zM18.9 100.08a1.08 1.08 0 110-2.16 1.08 1.08 0 010 2.16zm-2.52-4.68a1.08 1.08 0 10-2.161 0 1.08 1.08 0 002.161 0zm15.84 10.8a1.08 1.08 0 112.16.001 1.08 1.08 0 01-2.16-.001zm0-7.2a1.08 1.08 0 102.16 0 1.08 1.08 0 00-2.16 0zm-6.12 6.12a1.08 1.08 0 100 2.161 1.08 1.08 0 000-2.161zm0-7.2a1.08 1.08 0 110 2.161 1.08 1.08 0 010-2.161zm3.6-3.6a1.08 1.08 0 110 2.161 1.08 1.08 0 010-2.161zm1.08 8.28a1.08 1.08 0 10-2.16.001 1.08 1.08 0 002.16-.001zm-8.28-8.28a1.08 1.08 0 110 2.161 1.08 1.08 0 010-2.161zm1.08 8.28a1.08 1.08 0 10-2.16.001 1.08 1.08 0 002.16-.001zm23.04 3.6a1.08 1.08 0 112.16 0 1.08 1.08 0 01-2.16 0zm0-7.2a1.08 1.08 0 102.161 0 1.08 1.08 0 00-2.161 0zm-6.12 6.12a1.08 1.08 0 100 2.161 1.08 1.08 0 000-2.161zm0-7.2a1.08 1.08 0 110 2.161 1.08 1.08 0 010-2.161zm4.68-2.52a1.08 1.08 0 10-2.161 0 1.08 1.08 0 002.161 0zm-1.08 6.12a1.08 1.08 0 110 2.161 1.08 1.08 0 010-2.161zm-6.12-6.12a1.08 1.08 0 10-2.161 0 1.08 1.08 0 002.161 0zm-1.08 6.12a1.08 1.08 0 110 2.161 1.08 1.08 0 010-2.161zm18 3.6a1.08 1.08 0 100 2.161 1.08 1.08 0 000-2.161zm0-5.04a1.08 1.08 0 110-2.16 1.08 1.08 0 010 2.16zm2.52-4.68a1.08 1.08 0 102.16 0 1.08 1.08 0 00-2.16 0zm1.08 8.28a1.08 1.08 0 110-2.161 1.08 1.08 0 010 2.161zm-8.28-8.28a1.08 1.08 0 102.16 0 1.08 1.08 0 00-2.16 0zm1.08 8.28a1.08 1.08 0 110-2.161 1.08 1.08 0 010 2.161zm-19.06 9.52a60.04 60.04 0 001.323.672c.176.087.352.174.53.26a1.08 1.08 0 10-1.853-.932zm-2.54-4.48a1.08 1.08 0 110 2.161 1.08 1.08 0 010-2.161zm16.92 4.68a1.08 1.08 0 102.16 0 1.08 1.08 0 00-2.16 0zm-5.04 0a1.08 1.08 0 10-2.16.001 1.08 1.08 0 002.16-.001zm3.6-3.6a1.08 1.08 0 10-2.16.001 1.08 1.08 0 002.16-.001zm-1.08 6.12a1.08 1.08 0 01.441 2.066 65.45 65.45 0 01-1.327-.37 1.08 1.08 0 01.886-1.696zm-6.12-6.12a1.08 1.08 0 10-2.16.001 1.08 1.08 0 002.16-.001zm16.288 9.925c.177-.129.396-.205.632-.205h.002c.302 0 .573.123.77.323-.47-.034-.938-.073-1.404-.118zm-.448-6.325a1.08 1.08 0 102.16-.001 1.08 1.08 0 00-2.16.001zm5.76-3.6a1.08 1.08 0 10-2.16.001 1.08 1.08 0 002.16-.001zm-1.08 6.12a1.08 1.08 0 110 2.161 1.08 1.08 0 010-2.161zm-8.28-6.12a1.08 1.08 0 102.16-.001 1.08 1.08 0 00-2.16.001zm1.08 8.28a1.08 1.08 0 110-2.161 1.08 1.08 0 010 2.161zM0 60h60C60 26.864 33.138 0 0 0v60zm60 0c0 33.138 26.862 60 60 60V60H60z" fill="' . $color . '"/></g><defs><clipPath id="prefix__clip0_44_635"><path fill="#fff" d="M0 0h120v120H0z"/></clipPath></defs></svg>',
			],

			'bricks'       => [
				'default'  => '<svg width="42" height="44" viewBox="0 0 42 44" xmlns="http://www.w3.org/2000/svg"><g id="Page-1" fill="none" fill-rule="evenodd"><g id="brick-wall" fill="' . $color . '"><path d="M0 0h42v44H0V0zm1 1h40v20H1V1zM0 23h20v20H0V23zm22 0h20v20H22V23z"/></g></g></svg>',
				'inverted' => '<svg width="42" height="44" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M42 43V23H22v20h20zM0 23v20h20V23H0zM41 1H1v20h40V1z" fill="' . $color . '"/></svg>',
			],

			'circles'      => [
				'default'  => '<svg width="120" height="120" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M1.488 30c0 15.747 12.765 28.512 28.512 28.512S58.512 45.747 58.512 30 45.747 1.488 30 1.488 1.488 14.253 1.488 30zM30 60C13.431 60 0 46.569 0 30 0 13.431 13.431 0 30 0c16.569 0 30 13.431 30 30 0 16.569-13.431 30-30 30z" fill="' . $color . '"/><path fill-rule="evenodd" clip-rule="evenodd" d="M11.47 30c0 10.234 8.296 18.53 18.53 18.53 10.234 0 18.53-8.296 18.53-18.53 0-10.234-8.296-18.53-18.53-18.53-10.234 0-18.53 8.296-18.53 18.53zM30 50c-11.046 0-20-8.954-20-20s8.954-20 20-20 20 8.954 20 20-8.954 20-20 20zM1.488 90c0 15.747 12.765 28.512 28.512 28.512S58.512 105.747 58.512 90 45.747 61.488 30 61.488 1.488 74.253 1.488 90zM30 120c-16.569 0-30-13.431-30-30 0-16.569 13.431-30 30-30 16.569 0 30 13.431 30 30 0 16.569-13.431 30-30 30z" fill="' . $color . '"/><path fill-rule="evenodd" clip-rule="evenodd" d="M11.47 90c0 10.234 8.296 18.531 18.53 18.531 10.234 0 18.53-8.297 18.53-18.531 0-10.234-8.296-18.53-18.53-18.53-10.234 0-18.53 8.296-18.53 18.53zM30 110c-11.046 0-20-8.954-20-20s8.954-20 20-20 20 8.954 20 20-8.954 20-20 20zM61.488 30c0 15.747 12.765 28.512 28.512 28.512S118.512 45.747 118.512 30 105.747 1.488 90 1.488 61.488 14.253 61.488 30zM90 60c-16.569 0-30-13.431-30-30C60 13.431 73.431 0 90 0c16.569 0 30 13.431 30 30 0 16.569-13.431 30-30 30z" fill="' . $color . '"/><path fill-rule="evenodd" clip-rule="evenodd" d="M71.47 30c0 10.234 8.296 18.53 18.53 18.53 10.234 0 18.531-8.296 18.531-18.53 0-10.234-8.297-18.53-18.531-18.53-10.234 0-18.53 8.296-18.53 18.53zM90 50c-11.046 0-20-8.954-20-20s8.954-20 20-20 20 8.954 20 20-8.954 20-20 20zM61.488 90c0 15.747 12.765 28.512 28.512 28.512S118.512 105.747 118.512 90 105.747 61.488 90 61.488 61.488 74.253 61.488 90zM90 120c-16.569 0-30-13.431-30-30 0-16.569 13.431-30 30-30 16.569 0 30 13.431 30 30 0 16.569-13.431 30-30 30z" fill="' . $color . '"/><path fill-rule="evenodd" clip-rule="evenodd" d="M71.47 90c0 10.234 8.296 18.531 18.53 18.531 10.234 0 18.531-8.297 18.531-18.531 0-10.234-8.297-18.53-18.531-18.53-10.234 0-18.53 8.296-18.53 18.53zM90 110c-11.046 0-20-8.954-20-20s8.954-20 20-20 20 8.954 20 20-8.954 20-20 20z" fill="' . $color . '"/></svg>',
				'inverted' => '<svg width="120" height="120" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M120 0H0v120h120V0zm0 30c0 16.569-13.431 30-30 30-16.569 0-30-13.431-30-30C60 13.431 73.431 0 90 0c16.569 0 30 13.431 30 30zm0 60c0 16.569-13.431 30-30 30-16.564 0-29.993-13.425-30-29.987C59.993 106.575 46.564 120 30 120c-16.569 0-30-13.431-30-30 0-16.569 13.431-30 30-30 16.564 0 29.993 13.425 30 29.987C60.007 73.425 73.436 60 90 60c16.569 0 30 13.431 30 30zM0 30C0 13.431 13.431 0 30 0c16.569 0 30 13.431 30 30 0 16.569-13.431 30-30 30C13.431 60 0 46.569 0 30zm90 28.512c-15.747 0-28.512-12.765-28.512-28.512S74.253 1.488 90 1.488 118.512 14.253 118.512 30 105.747 58.512 90 58.512zM1.488 30c0 15.747 12.765 28.512 28.512 28.512S58.512 45.747 58.512 30 45.747 1.488 30 1.488 1.488 14.253 1.488 30zm9.981 0c0 10.234 8.297 18.53 18.531 18.53 10.234 0 18.53-8.296 18.53-18.53 0-10.234-8.296-18.53-18.53-18.53-10.234 0-18.53 8.296-18.53 18.53zM30 50c-11.046 0-20-8.954-20-20s8.954-20 20-20 20 8.954 20 20-8.954 20-20 20zm0 68.512c-15.747 0-28.512-12.765-28.512-28.512S14.253 61.488 30 61.488 58.512 74.253 58.512 90 45.747 118.512 30 118.512zM11.47 90c0 10.234 8.296 18.531 18.53 18.531 10.234 0 18.53-8.297 18.53-18.531 0-10.234-8.296-18.53-18.53-18.53-10.234 0-18.53 8.296-18.53 18.53zM30 110c-11.046 0-20-8.954-20-20s8.954-20 20-20 20 8.954 20 20-8.954 20-20 20zm41.47-80c0 10.234 8.296 18.53 18.53 18.53 10.234 0 18.531-8.296 18.531-18.53 0-10.234-8.297-18.53-18.531-18.53-10.234 0-18.53 8.296-18.53 18.53zM90 50c-11.046 0-20-8.954-20-20s8.954-20 20-20 20 8.954 20 20-8.954 20-20 20zm0 68.512c-15.747 0-28.512-12.765-28.512-28.512S74.253 61.488 90 61.488 118.512 74.253 118.512 90 105.747 118.512 90 118.512zM71.47 90c0 10.234 8.296 18.531 18.53 18.531 10.234 0 18.531-8.297 18.531-18.531 0-10.234-8.297-18.53-18.531-18.53-10.234 0-18.53 8.296-18.53 18.53zM90 110c-11.046 0-20-8.954-20-20s8.954-20 20-20 20 8.954 20 20-8.954 20-20 20z" fill="' . $color . '"/></svg>',
			],

			'dots'         => [
				'default'  => '<svg width="60" height="60" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M6 4a2 2 0 11-4 0 2 2 0 014 0" fill="' . $color . '"/></svg>',
				'inverted' => '<svg width="60" height="60" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M60 0H0v60h60V0zM5.414 5.414a2 2 0 10-2.828-2.828 2 2 0 002.828 2.828z" fill="' . $color . '"/></svg>',
			],

			'grid'         => [
				'default'  => '<svg width="40" height="40" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M20.5 20.5V40h-1V20.5H0v-1h19.5V0h1v19.5H40v1H20.5z" fill="' . $color . '"/></svg>',
				'inverted' => '<svg width="40" height="40" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M19.5 0H0v19.5h19.5V0zm1 0v19.5H40V0H20.5zM40 20.5H20.5V40H40V20.5zM19.5 40V20.5H0V40h19.5z" fill="' . $color . '"/></svg>',
			],

			'hexagon'      => [
				'default'  => '<svg width="28" height="49" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M13.99 9.25l13 7.5v15l-13 7.5L1 31.75v-15l12.99-7.5zM3 17.9v12.7l10.99 6.34 11-6.35V17.9l-11-6.34L3 17.9zM0 15l12.98-7.5V0h-2v6.35L0 12.69V15zm0 18.5L12.98 41v8h-2v-6.85L0 35.81V33.5zM15 0v7.5L27.99 15H28v-2.31h-.01L17 6.35V0h-2zm0 49v-8l12.99-7.5H28v2.31h-.01L17 42.15V49h-2z" fill="' . $color . '"/></svg>',
				'inverted' => '<svg width="28" height="49" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M10.98 0H0v12.69l10.98-6.34V0zm2 0v7.5L0 15v18.5L12.98 41v8H15v-8l12.99-7.5H28V15h-.01L15 7.5V0h-2.02zM17 0v6.35l10.99 6.34H28V0H17zm11 35.81h-.01L17 42.15V49h11V35.81zM10.98 49v-6.85L0 35.81V49h10.98zm16.01-32.25l-13-7.5L1 16.75v15l12.99 7.5 13-7.5v-15zM3 30.6V17.9l10.99-6.34 11 6.34v12.69l-11 6.35L3 30.6z" fill="' . $color . '"/></svg>',
			],

			'half-diamond' => [
				'default'  => '<svg width="80" height="120" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#prefix__clip0_43_320)" fill="' . $color . '"><path d="M40 0L0 30v30l40-30V0zm0 30v30l40 30V60L40 30zM40-30V0l40 30V0L40-30zm0 90L0 90v30l40-30V60zm0 30v30l40 30v-30L40 90z"/></g><defs><clipPath id="prefix__clip0_43_320"><path fill="#fff" d="M0 0h80v120H0z"/></clipPath></defs></svg>',
				'inverted' => '<svg width="80" height="120" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M40 0H0v30L40 0zM0 60v30l40-30v30L0 120h40V90l40 30V90L40 60V30l40 30V30L40 0v30L0 60z" fill="' . $color . '"/></svg>',
			],

			'half-circle'  => [
				'default'  => '<svg width="100" height="50" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#prefix__clip0_43_335)"><path fill-rule="evenodd" clip-rule="evenodd" d="M50 50c27.614 0 50-22.386 50-50v50H50zM0 0c0 27.614 22.386 50 50 50H0V0zm0 0c0-27.614 22.386-50 50-50s50 22.386 50 50H0z" fill="' . $color . '"/></g><defs><clipPath id="prefix__clip0_43_335"><path fill="#fff" d="M0 0h100v50H0z"/></clipPath></defs></svg>',
				'inverted' => '<svg width="100" height="50" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#prefix__clip0_43_332)"><path d="M50 50c27.614 0 50-22.386 50-50S77.614-50 50-50 0-27.614 0 0s22.386 50 50 50z" fill="' . $color . '"/></g><defs><clipPath id="prefix__clip0_43_332"><path fill="#fff" d="M0 0h100v50H0z"/></clipPath></defs></svg>',
			],

			'pastel'       => [
				'default'  => '<svg width="75" height="75" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#prefix__clip0_43_338)"><path d="M32.763-11.976c-1.05-.075-1.95.676-2.024 1.726L29.764.849c-.075 1.05.675 1.95 1.725 2.026 1.05.075 1.95-.675 2.025-1.725l.975-11.1c.075-1.05-.675-1.95-1.725-2.025l-.001-.001zM54.299 1.32a1.911 1.911 0 00-.386.015c-.975.15-1.725 1.05-1.575 2.1l1.5 11.025c.15.975 1.05 1.725 2.1 1.575a1.732 1.732 0 001.575-2.1l-1.5-11.025c-.131-.853-.836-1.533-1.714-1.59zM7.369 2.54a1.809 1.809 0 00-1.662 1.663c-.075 1.05.675 1.952 1.65 2.027l11.1 1.05c.975.15 1.95-.601 2.025-1.651.15-.975-.6-1.95-1.65-2.025l-11.1-1.05a1.642 1.642 0 00-.363-.015v.001zM1.76 13.017a1.824 1.824 0 00-1.285.6l-7.65 8.101c-.75.75-.675 1.95.075 2.625s1.95.674 2.625-.076l7.651-8.099c.75-.75.674-1.95-.076-2.625a1.786 1.786 0 00-1.34-.526zm75 0a1.824 1.824 0 00-1.285.6l-7.65 8.101c-.75.75-.675 1.95.075 2.625s1.95.674 2.625-.076l7.651-8.099c.75-.75.674-1.95-.076-2.625a1.786 1.786 0 00-1.34-.526zm-39.731 2.906a1.786 1.786 0 00-1.34.527l-7.95 7.723c-.75.675-.826 1.875-.076 2.625.675.75 1.875.752 2.625.077l7.95-7.725c.75-.675.826-1.875.076-2.625a1.827 1.827 0 00-1.285-.602zm24.639 18.928c-.24.02-.48.085-.705.197a1.903 1.903 0 00-.825 2.55l5.1 9.902a1.902 1.902 0 002.55.824c.975-.45 1.276-1.574.826-2.55l-5.1-9.9c-.395-.73-1.125-1.083-1.846-1.023zm-50.37-4.862c-.372 0-.734.117-1.035.336-.825.6-1.05 1.725-.524 2.625l6.15 9.223c.6.9 1.8 1.127 2.625.526.9-.6 1.124-1.8.524-2.624l-6.15-9.226a1.911 1.911 0 00-1.59-.86zm32.705 9.766c-.12-.006-.243 0-.365.019l-10.95 2.175c-1.05.15-1.725 1.126-1.5 2.176.15 1.05 1.126 1.725 2.176 1.5l10.95-2.175c1.05-.15 1.725-1.125 1.5-2.175a1.99 1.99 0 00-1.811-1.52zm4.556 12.195a1.933 1.933 0 00-1.845.949c-.45.9-.15 2.025.75 2.55l9.75 5.4c.9.45 2.025.15 2.55-.75.525-.9.15-2.025-.75-2.55l-9.75-5.4c-.22-.11-.46-.178-.705-.199zM71.913 58c-1.05-.075-1.875.748-1.95 1.798l-.45 11.1c-.075 1.05.75 1.876 1.8 1.95.975 0 1.875-.75 1.95-1.8l.45-11.1c.075-1.05-.75-1.873-1.8-1.948zm-55.44 1.08c-.38.031-.741.178-1.035.42l-8.775 6.825c-.75.6-.9 1.8-.3 2.625.6.75 1.8.9 2.626.3l8.775-6.827c.75-.6.9-1.8.3-2.625a1.783 1.783 0 00-1.591-.72v.002zm16.29 3.945c-1.05-.075-1.95.675-2.024 1.725l-.975 11.099c-.075 1.05.675 1.95 1.725 2.026 1.05.075 1.95-.675 2.025-1.725l.975-11.102c.075-1.05-.675-1.95-1.725-2.024l-.001.001z" fill="' . $color . '"/></g><defs><clipPath id="prefix__clip0_43_338"><path fill="#fff" d="M0 0h75v75H0z"/></clipPath></defs></svg>',
				'inverted' => '<svg width="75" height="75" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M29.839 0l-.075.849c-.075 1.05.675 1.95 1.725 2.026 1.05.075 1.95-.675 2.025-1.725L33.615 0H75v14.12l-7.175 7.598c-.75.75-.675 1.95.075 2.625s1.95.674 2.625-.076L75 19.53V75H33.615l.874-9.952c.075-1.05-.675-1.95-1.725-2.024l-.001.001c-1.05-.075-1.95.675-2.024 1.725l-.9 10.25H0V19.53l3.176-3.362c.75-.75.674-1.95-.076-2.625a1.786 1.786 0 00-1.34-.526 1.824 1.824 0 00-1.285.6L0 14.12V0h29.839zm24.074 1.335c.128-.018.257-.023.386-.015.878.057 1.583.737 1.714 1.59l1.5 11.025a1.732 1.732 0 01-1.575 2.1c-1.05.15-1.95-.6-2.1-1.575l-1.5-11.025c-.15-1.05.6-1.95 1.575-2.1zM6.231 3.065a1.809 1.809 0 011.138-.525v-.001c.121-.008.243-.003.363.015l11.1 1.05c1.05.075 1.8 1.05 1.65 2.025-.075 1.05-1.05 1.801-2.025 1.651l-11.1-1.05c-.975-.075-1.725-.977-1.65-2.027.033-.43.22-.833.524-1.138zm30.072 12.98a1.826 1.826 0 012.01.48c.751.75.675 1.95-.075 2.625l-7.95 7.725c-.75.675-1.95.673-2.625-.077-.75-.75-.674-1.95.076-2.625l7.95-7.723c.175-.176.384-.314.614-.405zm24.66 19.003c.225-.112.465-.177.705-.197.721-.06 1.451.293 1.846 1.023l5.1 9.9c.45.976.149 2.1-.826 2.55a1.902 1.902 0 01-2.55-.824l-5.1-9.902a1.903 1.903 0 01.825-2.55zm-50.7-4.723c.3-.219.663-.336 1.035-.336a1.913 1.913 0 011.59.86l6.15 9.226c.6.824.376 2.024-.524 2.624-.825.601-2.025.374-2.625-.526l-6.15-9.223c-.526-.9-.301-2.025.524-2.625zm33.375 9.449c.122-.019.245-.025.365-.019a1.99 1.99 0 011.811 1.52c.225 1.05-.45 2.025-1.5 2.175l-10.95 2.175c-1.05.225-2.026-.45-2.176-1.5-.225-1.05.45-2.026 1.5-2.176l10.95-2.175zm3.86 12.382a1.958 1.958 0 011.766-.007l9.75 5.4c.9.525 1.275 1.65.75 2.55-.525.9-1.65 1.2-2.55.75l-9.75-5.4c-.9-.525-1.2-1.65-.75-2.55.185-.317.458-.575.784-.743zm22.465 7.642c.075-1.05.9-1.873 1.95-1.798 1.05.075 1.875.898 1.8 1.948l-.45 11.1c-.075 1.05-.975 1.8-1.95 1.8-1.05-.074-1.875-.9-1.8-1.95l.45-11.1zM15.438 59.5a1.866 1.866 0 011.035-.42v-.002a1.783 1.783 0 011.591.72c.6.825.45 2.025-.3 2.625L8.989 69.25c-.826.6-2.026.45-2.626-.3-.6-.825-.45-2.025.3-2.625l8.775-6.825z" fill="' . $color . '"/></svg>',
			],

			'square'       => [
				'default'  => '<svg width="72" height="72" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M72 51.036v3.133H13.022l-3.334 2.96L0 57.178v-6.14h72zM72 42.13v3.132H22.794l-3.334 2.96L0 48.325v-6.195h72zM72 33.063v3.132H33.01l-3.334 2.961L0 39.305v-6.242h72zM72 24.156v3.132H42.78l-3.334 2.962L0 30.45v-6.295h72zM72 15.25v3.132H52.996l-3.334 2.96L0 21.593v-6.343h72zM72 6.343v3.132h-9.232l-3.334 2.961L0 12.738V6.343h72zM72 0v1.208l-2.499 2.22L0 3.779V0h72zM72 60.043v3.133H2.955L0 65.8v-5.756h72zM72 68.95H0V72h72v-3.05z" fill="' . $color . '"/></svg>',
				'inverted' => '<svg width="72" height="68" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M72 5.343V.208l-2.499 2.22L0 2.779v2.565h72zm0 3.132h-9.232l-3.334 2.961L0 11.738v2.511h72V8.475zm0 8.907H52.996l-3.334 2.96L0 20.593v2.564h72v-5.774zm0 8.906H42.78l-3.334 2.962L0 29.45v2.611h72v-5.774zm0 8.907H33.01l-3.334 2.961L0 38.305v2.824h72v-5.934zm0 9.067H22.794l-3.334 2.96L0 47.325v2.712h72v-5.774zm0 8.907H13.022l-3.334 2.96L0 56.178v2.866h72V53.17zm0 9.007H2.955L0 64.8v3.151h72v-5.774z" fill="' . $color . '"/></svg>',
			],

			'square-2'     => [
				'default'  => '<svg width="60" height="60" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M26 1h-7a3 3 0 00-3 3v7a3 3 0 003 3h7a3 3 0 003-3V4a3 3 0 00-3-3zM56 1h-7a3 3 0 00-3 3v7a3 3 0 003 3h7a3 3 0 003-3V4a3 3 0 00-3-3zM11 31H4a3 3 0 00-3 3v7a3 3 0 003 3h7a3 3 0 003-3v-7a3 3 0 00-3-3zM26 31h-7a3 3 0 00-3 3v7a3 3 0 003 3h7a3 3 0 003-3v-7a3 3 0 00-3-3zM41 31h-7a3 3 0 00-3 3v7a3 3 0 003 3h7a3 3 0 003-3v-7a3 3 0 00-3-3zM11 16H4a3 3 0 00-3 3v7a3 3 0 003 3h7a3 3 0 003-3v-7a3 3 0 00-3-3zM41 16h-7a3 3 0 00-3 3v7a3 3 0 003 3h7a3 3 0 003-3v-7a3 3 0 00-3-3zM56 16h-7a3 3 0 00-3 3v7a3 3 0 003 3h7a3 3 0 003-3v-7a3 3 0 00-3-3zM26 46h-7a3 3 0 00-3 3v7a3 3 0 003 3h7a3 3 0 003-3v-7a3 3 0 00-3-3zM56 46h-7a3 3 0 00-3 3v7a3 3 0 003 3h7a3 3 0 003-3v-7a3 3 0 00-3-3z" fill="' . $color . '"/></svg>',
				'inverted' => '<svg width="60" height="60" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M60 0H0v60h60V0zM19 46a3 3 0 00-3 3v7a3 3 0 003 3h7a3 3 0 003-3v-7a3 3 0 00-3-3h-7zM4 31a3 3 0 00-3 3v7a3 3 0 003 3h7a3 3 0 003-3v-7a3 3 0 00-3-3H4zM16 4a3 3 0 013-3h7a3 3 0 013 3v7a3 3 0 01-3 3h-7a3 3 0 01-3-3V4zm30 0a3 3 0 013-3h7a3 3 0 013 3v7a3 3 0 01-3 3h-7a3 3 0 01-3-3V4zM16 34a3 3 0 013-3h7a3 3 0 013 3v7a3 3 0 01-3 3h-7a3 3 0 01-3-3v-7zm18-3a3 3 0 00-3 3v7a3 3 0 003 3h7a3 3 0 003-3v-7a3 3 0 00-3-3h-7zM1 19a3 3 0 013-3h7a3 3 0 013 3v7a3 3 0 01-3 3H4a3 3 0 01-3-3v-7zm33-3a3 3 0 00-3 3v7a3 3 0 003 3h7a3 3 0 003-3v-7a3 3 0 00-3-3h-7zm12 3a3 3 0 013-3h7a3 3 0 013 3v7a3 3 0 01-3 3h-7a3 3 0 01-3-3v-7zm0 30a3 3 0 013-3h7a3 3 0 013 3v7a3 3 0 01-3 3h-7a3 3 0 01-3-3v-7z" fill="' . $color . '"/></svg>',
			],

			'triangle'     => [
				'default'  => '<svg width="100" height="100" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M0 0v100h1v-1l99-99H83L1 82V0H0zM98.178 85L100 83v17H83l15.178-15z" fill="' . $color . '"/><path d="M84 99H1v1h83v-1z" fill="' . $color . '"/></svg>',
				'inverted' => '<svg width="99" height="100" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M82 0H0v82L82 0zm17 83V0L0 99h83v.012L97.178 85 99 83z" fill="' . $color . '"/></svg>',
			],

			'triangle-2'   => [
				'default'  => '<svg width="60" height="60" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M30 30h30V0L45 15 30 30z" fill="' . $color . '" fill-opacity=".25"/><path d="M60 60H30V30l15 15 15 15z" fill="' . $color . '" fill-opacity=".75"/><path d="M0 60h30V30L15 45 0 60z" fill="' . $color . '" fill-opacity=".1"/><path d="M60 0H30v30l15-15L60 0z" fill="' . $color . '" fill-opacity=".05"/><path d="M30 30H0v30l15-15 15-15z" fill="' . $color . '" fill-opacity=".5"/><path d="M30 30H0V0l15 15 15 15z" fill="' . $color . '"/></svg>',
				'inverted' => '<svg width="60" height="60" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M30 30h30V0L45 15 30 30z" fill="' . $color . '" fill-opacity=".25"/><path d="M60 60H30V30l15 15 15 15z" fill="' . $color . '" fill-opacity=".75"/><path d="M0 60h30V30L15 45 0 60z" fill="' . $color . '" fill-opacity=".1"/><path d="M60 0H30v30l15-15L60 0z" fill="' . $color . '" fill-opacity=".05"/><path d="M30 30H0v30l15-15 15-15z" fill="' . $color . '" fill-opacity=".5"/><path d="M30 30H0V0l15 15 15 15z" fill="' . $color . '"/><path d="M0 0h30v30L15 15 0 0zM30 30h30v30L45 45 30 30z" fill="' . $color . '"/></svg>',
			],

			'triangle-3'   => [
				'default'  => '<svg width="60" height="60" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#prefix__clip0_44_683)" fill="' . $color . '"><path d="M28.48 45.76l-.127.063.254-.129-.127.066zM24.004 47.996l-.255.128.51-.257-.255.129z"/><path d="M60 30L28.607 45.695v-31.39L60 30zM28.352 14.178v31.644l-4.093 2.046V12.131l4.093 2.046zM23.75 11.875v36.25l-3.463 1.732V10.143l3.462 1.732zM19.268 9.635v40.73l-2.291 1.146V8.489l2.291 1.146zM15.449 7.726v44.548l-1.563.78V6.945l1.563.78zM11.849 5.927v48.146l-1.222.613V5.313l1.222.614zM8.08 4.044v51.912l-.804.406V3.638l.805.406zM4.221 2.112v55.775l-.464.234V1.878l.464.234zM.193.097v59.806L0 60V0l.193.097zM31.52 15.76l.127.063-.254-.129.127.066zM35.996 17.996l.255.128-.51-.256.255.128z"/><path d="M0 0l31.393 15.695v-31.39L0 0zM31.648-15.822v31.645l4.093 2.045v-35.737l-4.093 2.046zM36.25-18.125v36.25l3.463 1.732v-39.714l-3.462 1.732zM40.732-20.365v40.73l2.291 1.146v-43.022l-2.291 1.146zM44.551-22.274v44.548l1.563.78v-46.109l-1.563.78zM48.151-24.073v48.146l1.222.613v-49.373l-1.222.614zM51.92-25.956v51.912l.804.406v-52.724l-.805.406zM55.779-27.888v55.775l.464.234v-56.243l-.464.234zM59.807-29.903v59.806L60 30v-60l-.193.097zM0 60l31.393 15.695v-31.39L0 60zM31.648 44.178v31.645l4.093 2.045V42.131l-4.093 2.047zM36.25 41.875v36.25l3.463 1.732V40.143l-3.462 1.732zM40.732 39.635v40.73l2.291 1.146V38.489l-2.291 1.146zM44.551 37.726v44.548l1.563.78V36.945l-1.563.78zM48.151 35.927v48.146l1.222.613V35.313l-1.222.614zM51.92 34.044v51.912l.804.406V33.638l-.805.406zM55.779 32.112v55.775l.464.234V31.878l-.464.234zM59.807 30.097v59.806L60 90V30l-.193.097z"/></g><defs><clipPath id="prefix__clip0_44_683"><path fill="#fff" d="M0 0h60v60H0z"/></clipPath></defs></svg>',
				'inverted' => '<svg width="60" height="60" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#prefix__clip0_44_680)"><path fill-rule="evenodd" clip-rule="evenodd" d="M.193.097v59.806l3.564-1.782V1.878L0 0h31.393v15.694L3.786 1.894l.435.22v55.774l-.45.227 27.622-13.809V60H0l31.393 15.695.127.066.127.062-.254-.129V60h.255v15.823l4.093 2.045.51.256 3.462 1.733V60h1.019v20.365l2.291 1.146V60h1.528v22.274l1.563.78V60h2.037v24.073l1.222.613V60h2.546v25.956l.805.406V60h3.055v27.887l.464.234V60h3.564v29.903L60 90V60h-.193V30.097L60 30l-3.757 1.878V60h-.464V32.112l.436-.22-27.608 13.803v-31.39l27.621 13.809-.45-.227V0h.465v28.121l3.564 1.782V0H60v-30l-.193.097V0h-3.564v-28.122l-.464.234V0h-3.055v-26.362l-.805.406V0h-2.546v-24.686l-1.222.613V0h-2.037v-23.055l-1.563.78V0h-1.528v-21.511l-2.291 1.146V0h-1.019v-19.857l-3.462 1.732V0h-.51v-17.869l-4.093 2.046V0h-.255v-15.695L0 0l.193.097zm35.548 17.77v.001l-4.093-2.045V0h4.093v17.867zm.51.257l-.51-.257.255.129.255.128zm0 0V0h3.462v19.857l-3.462-1.733zM40.732 0v20.365l2.291 1.146V0h-2.291zm3.82 0v22.274l1.563.78V0H44.55zm3.6 0v24.073l1.221.613V0h-1.222zm3.767 0v25.956l.805.406V0h-.805zM28.607 45.694l-.127.066-.127.063.254-.129zM52.724 60V33.638l-.805.406V60h.805zm-3.35 0V35.313l-1.223.614V60h1.222zm-3.26 0V36.945l-1.563.78V60h1.563zm-3.09 0V38.489l-2.292 1.146V60h2.291zm-6.773 0v18.124l-.255-.128-.255-.129V60h.51zm0 0V41.875l3.462-1.732V60h-3.462zm-.51 0h-4.093V44.178l4.093-2.047V60zm-4.348-44.306l.254.128-.127-.061-.127-.067zM8.08 4.044v51.912l-.805.406V3.638l.805.406zm3.768 50.03V5.926l-1.222-.614v49.373l1.222-.613zm3.6-46.348v44.548l-1.563.78V6.945l1.563.78zm3.819 42.639V9.635L16.977 8.49V51.51l2.291-1.146zm4.482-38.49v36.25l-3.463 1.732V10.143l3.463 1.732zm.509 35.992l-.255.129-.255.128.51-.257zm0 0V12.131l4.093 2.046v31.645l-4.093 2.046z" fill="' . $color . '"/></g><defs><clipPath id="prefix__clip0_44_680"><path fill="#fff" d="M0 0h60v60H0z"/></clipPath></defs></svg>',
			],

			'wave'         => [
				'default'  => '<svg width="160" height="120" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#prefix__clip0_43_381)" stroke="' . $color . '" stroke-width="2"><path d="M-66.839 19.027C-44.46 18.538-22.38 7.377 0 7.5c22.383.123 57.617 15 80 15 22.383 0 57.617-14.877 80-15 22.381-.123 44.461 11.037 66.839 11.527M-66.839 49.028C-44.46 48.537-22.38 37.377 0 37.5c22.383.123 57.617 15 80 15 22.383 0 57.617-14.877 80-15 22.381-.123 44.461 11.037 66.839 11.528M-66.839 79.028C-44.46 78.537-22.38 67.377 0 67.5c22.383.123 57.617 15 80 15 22.383 0 57.617-14.877 80-15 22.381-.123 44.461 11.037 66.839 11.528M-66.839 109.027C-44.46 108.537-22.38 97.377 0 97.5c22.383.123 57.617 15 80 15 22.383 0 57.617-14.877 80-15 22.381-.123 44.461 11.037 66.839 11.527"/></g><defs><clipPath id="prefix__clip0_43_381"><path fill="#fff" d="M0 0h160v120H0z"/></clipPath></defs></svg>',
				'inverted' => '<svg width="160" height="120" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M160 0H0v6.5h.006c11.256.062 25.663 3.797 39.912 7.492l.333.086C54.701 17.825 68.965 21.5 80 21.5c11.035 0 25.299-3.675 39.749-7.422l.333-.086c14.249-3.695 28.656-7.43 39.913-7.492H160V0zm0 8.5c-11.005.062-25.218 3.747-39.624 7.482l-.125.032C105.892 19.737 91.347 23.5 80 23.5c-11.347 0-25.892-3.763-40.251-7.486l-.125-.032C25.218 12.247 11.005 8.562 0 8.5v28h.006c11.256.062 25.663 3.797 39.912 7.492l.333.086C54.701 47.825 68.965 51.5 80 51.5c11.035 0 25.299-3.675 39.749-7.422l.333-.086c14.249-3.695 28.656-7.43 39.913-7.492H160v-28zm0 30c-11.005.062-25.218 3.747-39.624 7.482l-.125.032C105.892 49.737 91.347 53.5 80 53.5c-11.347 0-25.892-3.763-40.251-7.486l-.125-.032C25.218 42.247 11.005 38.562 0 38.5v28h.006c11.256.062 25.663 3.797 39.912 7.492l.333.086C54.701 77.825 68.965 81.5 80 81.5c11.035 0 25.299-3.675 39.749-7.422l.333-.086c14.249-3.695 28.656-7.43 39.913-7.492H160v-28zm0 30c-11.005.062-25.218 3.747-39.624 7.482l-.125.032C105.892 79.737 91.347 83.5 80 83.5c-11.347 0-25.892-3.763-40.251-7.486l-.125-.032C25.218 72.247 11.005 68.562 0 68.5v28h.006c11.256.062 25.663 3.797 39.912 7.492l.333.086C54.701 107.825 68.965 111.5 80 111.5c11.035 0 25.299-3.675 39.749-7.422l.333-.086c14.249-3.695 28.656-7.43 39.913-7.492H160v-28zm0 30c-11.005.062-25.218 3.747-39.624 7.482l-.125.032C105.892 109.737 91.347 113.5 80 113.5c-11.347 0-25.892-3.763-40.251-7.486l-.125-.032C25.218 102.247 11.005 98.562 0 98.5V120h160V98.5z" fill="' . $color . '"/></svg>',
			],

			'x'            => [
				'default'  => '<svg width="40" height="40" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M0 38.59l2.83-2.83 1.41 1.41L1.41 40H0v-1.41zM0 1.4l2.83 2.83 1.41-1.41L1.41 0H0v1.41-.01zM38.59 40l-2.83-2.83 1.41-1.41L40 38.59V40h-1.41zM40 1.41l-2.83 2.83-1.41-1.41L38.59 0H40v1.41z" fill="' . $color . '"/><path d="M22.83 15.77L20 18.6v-.01l-2.83-2.83-1.41 1.41L18.59 20l-2.83 2.83 1.41 1.41L20 21.41l2.83 2.83 1.41-1.41L21.41 20l2.83-2.82-1.41-1.41z" fill="' . $color . '"/></svg>',
				'inverted' => '<svg width="40" height="40" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M1.41 0h37.18l-2.83 2.83 1.41 1.41L40 1.41v37.18l-2.83-2.83-1.41 1.41L38.59 40H1.41l2.83-2.83-1.41-1.41L0 38.59V1.4l2.83 2.83 1.41-1.41L1.41 0zM20 18.6l2.83-2.83 1.41 1.41L21.41 20l2.83 2.83-1.41 1.41L20 21.41l-2.83 2.83-1.41-1.41L18.59 20l-2.83-2.83 1.41-1.41L20 18.59v.01z" fill="' . $color . '"/></svg>',
			],
		];

		return isset( $patterns[ $name ][ $style ] ) ? $patterns[ $name ][ $style ] : '';
	}
}
